--[[---------------------------------------------------------------------------
	Chocolatier Two Market UI
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

local priceFont = { standardFontName, kEighteen, DefaultColor }
local smallPriceFont = { standardFontName, kEighteen, DefaultColor }
local countFont = { standardFontName, kSixteen, DefaultColor }

local self = gDialogTable.market
local char = gDialogTable.char or self.character[1]
local dialog = self.dialog

local allowHaggle = true
if self.noHaggle then allowHaggle = false end

gShopItems = gDialogTable.items
local itemCount = table.getn(gShopItems)

local transactionMade = false

local items = {}

-------------------------------------------------------------------------------

local insetX = 75
local insetY = 60
local windowX = ui.xOffset + insetX
local windowY = ui.yOffset + insetY
local windowW = ui.screenWidth - 2 * insetX
local windowH = 240

if itemCount > 30 then
	-- Expand to three matrix columns
	windowX = 80
	windowW = ui.screenWidth - windowX - 20
end

if itemCount > 45 then
	-- Shrink font to fit more
	priceFont[2] = 16
	countFont[2] = 14
end

if itemCount > 57 then
	-- Expand window downwards as needed to fit more matrix rows
	windowH = windowH + bsgFloor((itemCount - 57) / 3 + .9) * (bsgFontHeight(priceFont) + 2)
end

-- Make small price font even smaller...
smallPriceFont[2] = bsgFloor(priceFont[2] * .9)

local charX = windowX + 2
local charY = windowY - 50

local fieldX = charX + ui.charW
local fieldY = windowY
local fieldW = windowX + windowW - fieldX - ui.devButtonW
local fieldH = windowH

-- Make room for character dialog at the top
local dialogY = fieldY
local dialogH = 100
local dialogW = kMax - 5
if dialog then
	fieldY = fieldY + dialogH
	fieldH = fieldH - dialogH
end

-------------------------------------------------------------------------------
-- Price tracking

local function TrackPrices()
	for _,item in ipairs(gShopItems) do
		if (not item.lowseen) or item.price < item.lowseen then item.lowseen = item.price
		elseif (not item.highseen) or item.price > item.highseen then item.highseen = item.price
		end
		
		if not item.highseen then item.highseen = item.lowseen
		elseif not item.lowseen then item.lowseen = item.highseen
		end
	end
end

-------------------------------------------------------------------------------
-- Haggling

local function EnableHaggle()
	if allowHaggle then
		EnableWindow("haggle", not (char:IsAngry() or self.owned or gSim.rank==0))
	else
		EnableWindow("haggle", false)
	end
end

local function Haggle()
	local action = DisplayDialog { "ui/haggle.lua", market=self, char=char }
	EnableHaggle()
	local newColor = nil
	if action == "up" then newColor = WorsePriceColor
	elseif action == "down" then newColor = BetterPriceColor
	end
	if newColor then
		TrackPrices()
		for _,item in ipairs(gShopItems) do
			SetLabel(item.name.."_price", newColor..bsgDollars(item.price))
		end
	end
end

-------------------------------------------------------------------------------

local function CloseDialog()
	-- Track seen and port on items not in inventory
	for _,item in ipairs(gShopItems) do
		if item.inventory == 0 then
			item.lastprice = item.price
			item.lastport = self.port.name
		end
	end

	gShopItems = nil
	gMoreLabel = nil
	CloseWindow()
	if transactionMade then gSim:TickDay() end
end

-------------------------------------------------------------------------------

local function EnterNumber(name, x, y)
	local item = LItem:ByName(name)
	local count = DisplayDialog { "ui/buysell.lua", item=item }
	count = tonumber(count) or 0
	if count > 0 then
		local cost = count * item.price
		if cost <= gSim.money then
			PlaySound("buy")

			-- Once per week a transaction will raise this character's mood
			if char.moodWeek ~= gSim.weeks then
				char:SetMood(char.mood + LCharacter.kPurchaseDelta)
				char.moodWeek = gSim.weeks
			end
			
			local portName = GetString(self.port.name)
			
			transactionMade = true
			item.inventory = item.inventory + count
			item.lastprice = item.price
			item.lastport = self.port.name
			item.usetime = gSim.weeks
			gSim:QueueMessage(GetString("buy_notice", tostring(count), GetString(item.name), bsgDollars(item.price), portName))
			gSim:AdjustMoney(-cost)
			gSim:InventoryChanged()
			gSim:FlushMessages()
		else
			DisplayDialog { "ui/okdialog.lua", body="buy_lowcash" }
		end
	end
	EnableHaggle()
	
	-- Tutorial Special: During "tutor09" and "tutor11" auto-close dialog
	if count > 0 and gSim.rank == 0 and gSim.quest and (gSim.quest.name == "tutor09" or gSim.quest.name == "tutor11") then
		CloseDialog()
	end
end

-------------------------------------------------------------------------------
-- lay out items as a bunch of sacks

local function LayoutSacks()
	local itemWidth = ui.sackWidth + 5
	local itemHeight = ui.sackHeight + 37 + bsgFontHeight(countFont) + 5
	local itemsPerRow = bsgFloor(fieldW / itemWidth)
	
	-- Force two rows with more than four sacks
	if itemCount > 4 and itemCount <= itemsPerRow then
		itemsPerRow = itemsPerRow / 2
	end

	-- Determine how many rows and how many items to place on each row
	local colCount = {}
	local rowCount = 1
	if itemCount > itemsPerRow then
		local total = itemCount
		rowCount = bsgFloor((itemCount + itemsPerRow - 1) / itemsPerRow)
		for i=1,rowCount do
			local count = bsgFloor((itemCount + rowCount - 1) / rowCount)
			if count > total then count = total end
			colCount[i] = count
			total = total - count
		end
	else
		colCount[1] = itemCount
	end

	local y = fieldY - windowY + (fieldH - itemHeight * rowCount) / 2

	local n = 0
	local row = 1
	local x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2

	local offsetX = (ui.sackWidth - ui.itemWidth) / 2
	local offsetY = (ui.sackHeight - ui.itemHeight) / 2
	
	local seasonWeek = bsgMod(gSim.weeks, 52)
	for _,item in ipairs(gShopItems) do
		local gx = windowX + x - 67
		local gy = windowY + y + 67
		local iname = item.name
		local font = priceFont
		if item.price > 9999 then font = smallPriceFont end
		
		local sack = "image/icon_sack"
		if item.startWeek < 0 or item.endWeek < 52 then
			if item:InSeason(seasonWeek) then sack = "image/icon_sack_inseason"
			else sack = "image/icon_sack_outseason"
			end
		end
		
		table.insert(items,
			Rollover
			{
				x=x,y=y,w=itemWidth,h=itemHeight, fit=true,
				contents=item:RolloverTarget("click_buy"),
				command=function() EnterNumber(iname,gx,gy) end,
				
				Bitmap { x=kCenter,y=0, image="image/pricetag", },
				Text { x=kCenter,y=3,w=54,h=20, font=font, name=item.name.."_price", label="#"..bsgDollars(item.price) },
				Bitmap { x=0,y=37, image=sack,
					Bitmap { x=offsetX,y=offsetY, image="item/"..iname }
				},
			})

		x = x + itemWidth
		n = n + 1
		if n == colCount[row] and row < rowCount then
			n = 0
			row = row + 1
			y = y + itemHeight
			x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2
		end
	end
end

-------------------------------------------------------------------------------
--

LayoutSacks()

TrackPrices()

if dialog then dialog = LabelString(dialog) end

local buttonX = windowW - ui.devButtonW
local buttonY = charY + ui.charH - windowY - 80		--100
MakeDialog
{
	name=self.name,
	DialogLedger(),
	Frame
	{
		x=windowX,y=windowY,w=windowW,h=windowH,fit=true, color=MarketColor,
		
		Text { x=fieldX-windowX,y=dialogY-windowY, w=dialogW,h=dialogH, label=dialog },

		Group(items),

		TextButton { x=buttonX,y=buttonY+40, name="haggle", label="haggle", command=Haggle, },
		
		TextButton { x=buttonX,y=buttonY+80, name="help", label="help", graphics=SmallButtonGraphics, command=function() HelpDialog(4) end },
		TextButton { x=buttonX+62,y=buttonY+80, name="ok", label="close", command=CloseDialog, graphics=MediumButtonGraphics },
	},
	BuildingBanner(self),
	char:Visual(windowX, charY, charNameFontDark),
}

EnableHaggle()
